

#
# AT&T Proprietary (Internal Use Only)
# All rights reserved
# This code is protected by copyright and distributed under licenses
# restricting its use, copying, distribution, and decompilation. It may
# only be reproduced or used under license from AT&T or its
# authorised licensees.
#

"""Creates USMA Sensor for GCP."""

IMAGE_VERSION = "usm-saas-sensor-gcp-9-13-55-9-1-49-master-ubuntu-22-04-5-lts"
IMAGE_PROJECT_ID = "att-cybersecurity"

DATA_VOLUME_SIZE = 128
NETWORK_TAG = 'usma-sensor'


SENSOR_NAME = ''
ZONE = ''
REGION = ''
PROJECT_ID = ''
DATA_VOLUME = ''
MACHINE_TYPE = ''
SSH_KEY = ''
NETWORK = ''
SUBNET_VALUE = ''
SUBNETWORK = ''
FIREWALL_RULES = ''
IP_RANGES = ''
PUBLIC_IP = True
SERVICE_ACCOUNT_MAIL = ''
ENVIRONMENT = ''
ALLOW_TCP_PORT = {}
ALLOW_UDP_PORT = {}


def GenerateConfig(context):
    global SENSOR_NAME
    global ZONE
    global REGION
    global PROJECT_ID
    global DATA_VOLUME
    global MACHINE_TYPE
    global SSH_KEY
    global NETWORK
    global SUBNET_VALUE
    global SUBNETWORK
    global FIREWALL_RULES
    global IP_RANGES
    global PUBLIC_IP
    global SERVICE_ACCOUNT_MAIL
    global ENVIRONMENT
    global ALLOW_TCP_PORT
    global ALLOW_UDP_PORT

    SENSOR_NAME = context.env.get('deployment', context.env.get('name', ''))
    ZONE = context.properties['zone']
    REGION = ZONE.split('-')[0] + '-' + ZONE.split('-')[1]
    PROJECT_ID = context.env["project"]
    DATA_VOLUME = '%s-data-volume' % SENSOR_NAME
    SSH_KEY = context.properties['ssh_key']
    NETWORK = "projects/%s/global/networks/%s" % (PROJECT_ID, context.properties['network'])
    SUBNET_VALUE = context.properties.get('subnetwork', '')
    SUBNETWORK = "projects/%s/regions/%s/subnetworks/%s" % (PROJECT_ID, REGION, SUBNET_VALUE) if SUBNET_VALUE else ''
    FIREWALL_RULES = 'allow-%s-traffic' % SENSOR_NAME
    IP_RANGES = context.properties['ip_ranges']
    PUBLIC_IP = context.properties['public_ip']
    SERVICE_ACCOUNT_MAIL = context.properties['service_account']
    MACHINE_TYPE = context.properties.get('machine_type', 'n2-standard-2')
    ENVIRONMENT = context.properties.get('environment', 'prod-gov')
    ALLOW_TCP_PORT[22] = context.properties['SSH']
    ALLOW_TCP_PORT[80] = context.properties['HTTP']
    ALLOW_TCP_PORT[601] = context.properties['SyslogTCP']
    ALLOW_TCP_PORT[602] = context.properties['SyslogTCPRFC5424']
    ALLOW_TCP_PORT[6514] = context.properties['SyslogTLS']
    ALLOW_TCP_PORT[6515] = context.properties['SyslogTLSRFC5424']
    ALLOW_TCP_PORT[5986] = context.properties['NXLog']
    ALLOW_TCP_PORT[5987] = context.properties['SyslogNGWEC']
    ALLOW_TCP_PORT[5005] = context.properties.get('RemoteDebugging', False) # Only for dev_template
    ALLOW_TCP_PORT[5006] = context.properties.get('Profiler', False) # Only for dev_template
    ALLOW_UDP_PORT[514] = context.properties['SyslogUDP']
    ALLOW_UDP_PORT[12201] = context.properties['Graylog']



    resources = [
        Volumes(context),
        Firewall(context),
        Instance(context)
    ]

    outputs = Outputs(context)

    return {'resources': resources, 'outputs': outputs}


def Firewall(context):
    TCP_PORTS = []
    UDP_PORTS = []
    for port in ALLOW_TCP_PORT:
        if ALLOW_TCP_PORT[port]:
            TCP_PORTS.append(port)
    for port in ALLOW_UDP_PORT:
        if ALLOW_UDP_PORT[port]:
            UDP_PORTS.append(port)
    return {
        'type': 'compute.v1.firewall',
        'name': FIREWALL_RULES,
        'properties': {
            'network': NETWORK,
            'sourceRanges': [IP_RANGES],
            'targetTags': [NETWORK_TAG],
            'allowed': [
                {
                    'IPProtocol': 'tcp',
                    'ports': TCP_PORTS
                },
                {
                    'IPProtocol': 'udp',
                    'ports': UDP_PORTS
            }
            ]
        }
    }


def Volumes(context):
    # Create a blank disk
    return {
        'type': 'compute.v1.disk',
        'name': DATA_VOLUME,
        'properties': {
            'sizeGb': DATA_VOLUME_SIZE,
            'zone': ZONE
        }
    }


def Instance(context):

    if PUBLIC_IP:
        admin_iface = {
            "network": NETWORK,
            'accessConfigs': [{
                'name': 'External NAT',
                'type': 'ONE_TO_ONE_NAT'
            }]
        }
    else:
        admin_iface = {
            "network": NETWORK
        }

    if SUBNETWORK:
        admin_iface["subnetwork"] = SUBNETWORK

    admin_iface["name"] = 'ens4'

    instance = {
        'name': SENSOR_NAME,
        'type': 'compute.v1.instance',
        'properties': {
            'zone': ZONE,
            'machineType': "zones/%s/machineTypes/%s" % (ZONE, MACHINE_TYPE),
            'disks': [
                {
                    'deviceName': 'boot',
                    'type': 'PERSISTENT',
                    'boot': True,
                    'autoDelete': True,
                    'initializeParams': {
                        'sourceImage': 'projects/%s/global/images/%s' % (IMAGE_PROJECT_ID, IMAGE_VERSION)
                    }
                },
                {
                    'deviceName': 'data',
                    'type': 'PERSISTENT',
                    'boot': False,
                    'autoDelete': False,
                    'source': '$(ref.%s.selfLink)' % DATA_VOLUME
                }
            ],
            'networkInterfaces': [admin_iface],
            "tags": {
                "items": [NETWORK_TAG]
            },
            'serviceAccounts':  [
              {
                'email': SERVICE_ACCOUNT_MAIL,
                'scopes': [
                  'https://www.googleapis.com/auth/cloud-platform',
                  'https://www.googleapis.com/auth/userinfo.email',
                  'https://www.googleapis.com/auth/compute.readonly',
                  'https://www.googleapis.com/auth/pubsub',
                  'https://www.googleapis.com/auth/admin.directory.user.readonly'
                ]
              }
            ],
            "metadata": {
                "kind": "compute#metadata",
                "items": [
                    {
                        "key": "av_profile",
                        "value": "sensor"
                    },
                    {
                        "key": "av_resources",
                        "value": "wyns"
                    },
                    {
                        "key": "environment",
                        "value": ENVIRONMENT
                    },
                    {
                        "key": "nodeName",
                        "value": SENSOR_NAME
                    },
                    {
                        "key": "ssh-keys",
                        "value": 'ubuntu:%s' % SSH_KEY
                    }
                ]
            }
        }
    }

    return instance


def Outputs(context):
    outputs = [
    {
        'name': 'CLIUser',
        'value': 'sysadmin'
    }]

    if PUBLIC_IP:
        outputs.append({
            'name': 'URL',
            'value': 'http://$(ref.%s.networkInterfaces[0].accessConfigs[0].natIP)/' % SENSOR_NAME
        })
    else:
        outputs.append({
            'name': 'URL',
            'value': "http://$(ref.%s.networkInterfaces[0].networkIP)/" % SENSOR_NAME
        })

    return outputs
